%=========================================================================
%
%  "Time Preference and the Distributions of Wealth and Income"
%   By Richard M. H. Suen 
%   Economic Inquiry (forthcoming)
%
%   This program is used to generate the benchmark results reported in 
%   Table 3 Panel A of the paper. 
%
%=========================================================================


close all
clear all

pause on

%=======================================
%
%  Assign parameter values.
%
%=======================================

alpha   = 0.33;

theta   = [0.005         0.010         0.01202      0.025        0.050       0.10       0.50];
delta_k = [0.0798079939  0.0799360984  0.080044917  0.081429784  0.08555056  0.0948155  0.1517206]; 

sigma = 1;
gamma   = 1.022;

N = 1000;
beta_max = 0.992;
beta_min = 0.966;

beta = linspace(beta_min, beta_max, N);

eps  = 0.939; 
vsig = 0.871;
vphi = 1;

delta_h = 0.037; 
delta_k_hat = gamma - 1 + delta_k; 

r_bar = ( gamma/beta_max + theta*gamma )/(1 + theta) - (1 - delta_k);


%==============================================================================
%
%  Distributions of labor hours, human capital and wealth
%
%==============================================================================

r_eq = 0.11;
Kd = (alpha/r_eq)^(1/(1 - alpha));
K_Y_Ratio = Kd^(1 - alpha);

w = (1 - alpha)*(Kd^alpha);

%==============================================================================

M = length(theta);

Results = zeros(M, 11);
Errors  = zeros(M, 1);

P = (((gamma^(sigma - 1))./beta - (1 - delta_h))/delta_h - vsig )/eps;
l = P./(1 + P);
h = ( ( vphi/delta_h )*(1 - l).^eps ).^(1/(1 - vsig));

Capital = zeros(M, N);

earn = w*(l.*h);

for i = 1:M

    D = ((( gamma^sigma )./beta - (1 - delta_k(i)) - r_eq )/theta(i)).^(1/sigma);
    
    k = earn./( D - r_eq + delta_k_hat(i) );
    
    if ( sum(k > 0) < N)
        
        fprintf('Error: Capital cannot be negative!')
        theta(i)
        
    end
    
    Capital(i,:) = k;
    
    c = D.*k;
    
    inc = earn + r_eq*k;
    
    Res = earn + (r_eq - delta_k_hat(i))*k - c;
    Check = sum(Res);
    Errors(i) = Kd - sum(k)/sum(l.*h); % Errors in the capital market clearing condition.
    
    
    %=================================================================================
    % 
    %  Summary Statistics for the Distributions
    %
    %=================================================================================
    
    
    s1 = l/sum(l);
    s2 = earn/sum(earn);
    s3 = k/sum(k);
    s4 = inc/sum(inc);
    
    ss1 = sort(s1);
    ss2 = sort(s2); 
    ss3 = sort(s3);
    ss4 = sort(s4); 
    
    share_l   = [0 cumsum(ss1)];
    share_e   = [0 cumsum(ss2)];
    share_k   = [0 cumsum(ss3)];
    share_in  = [0 cumsum(ss4)];
    
    Area_l  = 0;
    Area_e  = 0;
    Area_k  = 0;
    Area_in = 0;
    
    
    for j = 1:N
        
        Area_l  = Area_l  + ( share_l(j+1)  + share_l(j) )/N/2;
        Area_e  = Area_e  + ( share_e(j+1)  + share_e(j) )/N/2;
        Area_k  = Area_k  + ( share_k(j+1)  + share_k(j) )/N/2;
        Area_in = Area_in + ( share_in(j+1) + share_in(j) )/N/2;
        
    end
    
    Gini_l  = 1 - Area_l/0.5;
    Gini_e  = 1 - Area_e/0.5;
    Gini_in = 1 - Area_in/0.5;
    Gini_k  = 1 - Area_k/0.5;
    
    mean_l = sum(l)/N;
    var_l  = sum( ( l - mean_l ).^2 )/(N-1);
    std_l  = sqrt(var_l);
    CV_l   = std_l/mean_l;
    skew_l = sum(( ( l - mean_l )/std_l ).^3)/(N-1);
    
    mean_e = sum(earn)/N;
    var_e  = sum( ( earn - mean_e ).^2 )/(N-1);
    std_e  = sqrt(var_e);
    CV_e   = std_e/mean_e;
    skew_e = sum(( ( earn - mean_e )/std_e ).^3)/(N-1);
    
    mean_k = sum(k)/N;
    var_k  = sum( ( k - mean_k ).^2 )/(N-1);
    std_k  = sqrt(var_k);
    CV_k   = std_k/mean_k;
    skew_k = sum(( ( k - mean_k )/std_k ).^3)/(N-1);
    
    mean_in = sum(inc)/N;
    var_in  = sum( ( inc - mean_in ).^2 )/(N-1);
    std_in  = sqrt(var_in);
    CV_in   = std_in/mean_in;
    skew_in = sum(( ( inc - mean_in )/std_in ).^3)/(N-1);
    
    Corr_k_in = sum(( k - mean_k ).*( inc - mean_in ))/N/std_k/std_in;
    Corr_e_in = sum(( earn - mean_e ).*( inc - mean_in ))/N/std_e/std_in;
    Corr_k_e = sum(( k - mean_k ).*( earn - mean_e ))/N/std_k/std_e;
    
    S40 = share_k(400);
    S90 = 1 - share_k(901);
    S95 = 1 - share_k(951);
    S99 = 1 - share_k(991);
    
    T40 = share_in(400);
    T90 = 1 - share_in(901);
    T95 = 1 - share_in(951);
    T99 = 1 - share_in(991);
    
    Source = (earn)./(earn + r_eq*k);
    L1 = mean(Source(1:10));
    L2 = mean(Source(11:50));
    L3 = mean(Source(51:100));
    L4 = mean(Source(401:600));
    L5 = mean(Source(601:800));
    L6 = mean(Source(901:950));
    L7 = mean(Source(951:990));
    L8 = mean(Source(991:1000));
    
    %[theta(i) L1 L2 L3 L4 L5 L6 L7 L8]
    
    Results(i,:) = [theta(i) Gini_in Gini_k T40 T90 T95 T99 S40 S90 S95 S99];
    
end


%==============================================
%  Results reported in Table 3 Panel A
%==============================================

Results


 
